package Musings::Stylesheet;

use Musings::DB;
use Carp qw(croak);
use strict;

=head1 NAME

Musings::Stylesheet

=head1 SYNOPSIS

  use Musings::Stylesheet;
  my $sheet = Musings::Stylesheet->new;
  $sheet->stylesheet_name('tei2xhtml');
  $sheet->stylesheet_note('transform TEI to XHTML');
  $sheet->stylesheet_content('<?xml version='1.0'?><xsl:stylesheet...');
  $sheet->commit;
  my $id = $sheet->stylesheet_id;
  $sheet = Musings::Stylesheet->new(id => $id);
  my $stylesheet_name    = $sheet->stylesheet_name;
  my $stylesheet_note    = $sheet->stylesheet_note;
  my $stylesheet_content = $sheet->stylesheet_content;
  $sheet->delete;


=head1 DESCRIPTION

This package does the necessary I/O against the underlying database's stylesheets table.


=head1 METHODS


=head2 new

Creates a new stylesheet object. If passed an id parameter plus a valid stylesheet ID, then a specific stylesheet object will be created.


=head2 stylesheet_name

Get and set the name of the stylesheet.


=head2 stylesheet_note

Get and set the stylesheet's short descriptive blurb.


=head2 stylesheet_content

Get and set the entire content of the stylesheet. The value is expected to be an XSLT file.


=head2 commit

Save the current stylesheet object's attributes to the database. Returns true upon success.


=head2 delete

Deletes (drops) the current stylesheet from the database. Be careful. Your application is responsible for passing the ACID test.


=head2 get_stylesheets

Returns an array of all the stylesheet objects in the database. The returned objects will be sorted by stylesheet_name.


=head1 AUTHORS

Eric Lease Morgan <eric_morgan@infomotions.com>

=cut


sub new {

	# declare local variables
	my ($class, %opts) = @_;
	my $self = {};

	# check for an id
	if ($opts{id}) {
	
		my $dbh = Musings::DB->dbh();
		my $rv = $dbh->selectrow_hashref('SELECT * FROM stylesheets WHERE stylesheet_id = ?', undef, $opts{id});
		if (ref($rv) eq "HASH") { $self = $rv }
		else { return }
		
	}
	
	elsif ($opts{name}) {
	
		my $dbh = Musings::DB->dbh();
		my $rv = $dbh->selectrow_hashref('SELECT * FROM stylesheets WHERE stylesheet_id = ?', undef, $opts{name});
		if (ref($rv) eq "HASH") { $self = $rv }
		else { return } 
		
	}
	
	# return the object
	return bless $self, $class;
	
}


sub stylesheet_id {

	my $self = shift;
	return $self->{stylesheet_id};

}


sub stylesheet_name {

	# declare local variables
	my ($self, $stylesheet_name) = @_;
	
	# check for the existance of a note 
	if ($stylesheet_name) { $self->{stylesheet_name} = $stylesheet_name }
	
	# return the name
	return $self->{stylesheet_name};
	
}

sub stylesheet_note {

	# declare local variables
	my ($self, $stylesheet_note) = @_;
	
	# check for the existance of a note 
	if ($stylesheet_note) { $self->{stylesheet_note} = $stylesheet_note }
	
	# return the name
	return $self->{stylesheet_note};
	
}


sub stylesheet_content {

	# declare local variables
	my ($self, $stylesheet_content) = @_;
	
	# check for the existance of a note 
	if ($stylesheet_content) { $self->{stylesheet_content} = $stylesheet_content }
	
	# return the name
	return $self->{stylesheet_content};
	
}



sub commit {

	# get myself, :-)
	my $self = shift;
	
	# get a database handle
	my $dbh = Musings::DB->dbh();	
	
	# see if the object has an id
	if ($self->stylesheet_id()) {
	
		# update the record with this id
		my $return = $dbh->do('UPDATE stylesheets SET stylesheet_name = ?, stylesheet_note = ?, stylesheet_content = ? WHERE stylesheet_id = ?', undef, $self->stylesheet_name(), $self->stylesheet_note(), $self->stylesheet_content(), $self->stylesheet_id());
		if ($return > 1 || ! $return) { croak "Stylesheet update in commit() failed. $return records were updated." }
	
	}
	
	else {
	
		# get a new sequence
		my $id = Musings::DB->nextID();
		
		# create a new record
		my $return = $dbh->do('INSERT INTO stylesheets (stylesheet_id, stylesheet_name, stylesheet_note, stylesheet_content) VALUES (?, ?, ?, ?)', undef, $id, $self->stylesheet_name, $self->stylesheet_note, $self->stylesheet_content);
		if ($return > 1 || ! $return) { croak 'Stylesheet commit() failed.'; }
		$self->{stylesheet_id} = $id;
		
	}
	
	# done
	return 1;
	
}


sub delete {

	my $self = shift;

	if ($self->{stylesheet_id}) {

		my $dbh = Musings::DB->dbh();

		# now, delete the primary facet record
		my $rv = $dbh->do('DELETE FROM stylesheets WHERE stylesheet_id = ?', undef, $self->{stylesheet_id});
		if ($rv != 1) {croak ("Error deleting stylesheet record. Deleted $rv records.");}
		return 1;

	}

	return 0;

}


sub get_stylesheets {

	my $self = shift;
	my %opts = @_;
	my @rv   = ();

	# create and execute a query
	my $dbh = Musings::DB->dbh();

	my $stylesheet_ids = $dbh->selectcol_arrayref('SELECT stylesheet_id FROM stylesheets ORDER BY stylesheet_name');
			
	foreach my $stylesheet_id (@$stylesheet_ids) { push (@rv, Musings::Stylesheet->new(id => $stylesheet_id)) }	
	
	return @rv;
}


# return true
1;
